<?php
namespace Trs\Woocommerce;

use Exception;
use InvalidArgumentException;
use Trs\Factory\Registries\GlobalRegistry;
use Trs\Mapping\Interfaces\IReader;
use TrsVendors\Dgm\Shengine\Interfaces\IRule;
use TrsVendors\Dgm\Shengine\Units;
use TrsVendors\Dgm\Shengine\Woocommerce\Converters\PackageConverter;
use TrsVendors\Dgm\Shengine\Woocommerce\Converters\RateConverter;
use TrsVendors\Dgm\WcTools\WcTools;
use WC_Shipping_Method;


class ShippingMethod extends WC_Shipping_Method
{
    /** @noinspection PhpMissingParentConstructorInspection */
    public function __construct($instance_id = 0)
    {
        $this->id = 'tree_table_rate';
        $this->title = $this->method_title = 'Tree Table Rate';
        $this->instance_id = absint($instance_id);

        $this->supports = array(
            'settings',
            'shipping-zones',
            'instance-settings',
            'global-instance',
        );

        $this->init();
    }

    public function calculate_shipping($_package = array())
    {
        $globals = self::initGlobalRegistry(true);

        $rule = $this->loadRule($globals->reader);
        if (!isset($rule)) {
            return;
        }

        $cart = apply_filters('trs_include_non_shippable_items', false) ? WC()->cart : null;
        $package = PackageConverter::fromWoocommerceToCore($_package, $cart);
        $rates = $globals->processor->process(array($rule), $package);

        $_rates = RateConverter::fromCoreToWoocommerce(
            $rates,
            $this->title,
            join(':', array_filter(array($this->id, @$this->instance_id))).':'
        );

        foreach ($_rates as $_rate) {
            $this->add_rate($_rate);
        }
    }

    public function init()
    {
        $this->init_form_fields();
        $this->init_settings();

        $this->enabled = $this->get_option('enabled');
        $this->tax_status = $this->get_option('tax_status');
        $this->title = $this->get_option('label') ?: 'Tree Table Rate';
    }

    public function init_form_fields()
    {
        $meta = array(
            'enabled'    => array(
                'title'   => 'Enable/Disable',
                'type'    => 'checkbox',
                'label'   => 'Enable this shipping method',
                'default' => 'yes',
            ),
            'tax_status' => array(
                'title' 		=> 'Tax Status',
                'type' 			=> 'select',
                'class'         => 'wc-enhanced-select',
                'default' 		=> 'taxable',
                'options'		=> array(
                    'taxable' 	=> 'Taxable',
                    'none' 		=> 'Not taxable',
                ),
            ),
        );

        $rules = array(
            'rule' => array(
                'type' => 'rule',
                'default' => null,
            ),
        );

        $this->form_fields = $meta + $rules;

        $this->instance_form_fields =
            $meta +
            array(
                'label' => array(
                    'title' => 'Label',
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => 'Label in shipping zone table',
                    'css' => 'width: 15.7em',
                ),
            ) +
            $rules;

        unset($this->instance_form_fields['enabled']);
    }

    public function generate_rule_html()
    {
        ob_start();
        ?>
                <?php echo $this->generate_hidden_html('rule', array()) ?>
            </table>

            <?php include(__DIR__.'/../../tpl.php'); ?>

            <table>
        <?php
        return ob_get_clean();
    }

    public function generate_hidden_html($field, $definition)
    {
        $definition['type'] = 'hidden';
        $html = $this->generate_text_html($field, $definition);
        $html = preg_replace('/'.preg_quote('<tr', '/').'/', '<tr style="display:none"', $html, 1);
        return $html;
    }

    public function admin_options()
    {
        $methodTitleBkp = $this->method_title;
        $this->method_title .= ' Shipping';

        try {

            parent::admin_options();

        } catch (Exception $e){
            $this->method_title = $methodTitleBkp;
            /** @noinspection PhpUnhandledExceptionInspection */
            throw $e;
        }

        $this->method_title = $methodTitleBkp;
    }

    /**
     * @param mixed $config
     * @throws InvalidArgumentException On validation errors.
     */
    public function updateConfig($config)
    {
        $optionKey = $this->instance_id ? $this->get_instance_option_key() : $this->get_option_key();

        if (!is_array($config)) {
            throw new InvalidArgumentException('$config must be an array.');
        }

        if ($this->instance_id) {
            unset($config['enabled']);
        } else {
            $config['enabled'] = WcTools::bool2YesNo(isset($config['enabled']) ? $config['enabled'] : true);
        }

        $rule = null;
        if (!isset($config['rule']) || !is_array($rule = $config['rule'])) {
            throw new InvalidArgumentException('$config[\'rule\'] must be an array.');
        }

        $globals = self::initGlobalRegistry(false);

        try {
            $this->loadRule($globals->reader, $rule);
        } catch (Exception $e) {
            throw new InvalidArgumentException("Configuration validation failed. {$e->getMessage()}", 0, $e);
        }

        $config['rule'] = json_encode($config['rule']);
        $updated = update_option($optionKey, $config);
        if ($updated) {
            WcTools::purgeWoocommerceShippingCache();
        }
    }

    public function get_option($key, $empty_value = null) {

        $result = $empty_value;

        if (version_compare(WC()->version, '2.6', '>=') && empty($this->instance_id)) {

            add_filter(
                $filter = "woocommerce_shipping_instance_form_fields_{$this->id}",
                $stub = function() { return array(); }
            );

            $exception = null;
            try {
                $result = parent::get_option($key, $empty_value);
            }
            catch (Exception $e) {
                $exception = $e;
            }

            remove_filter($filter, $stub);

            if (isset($exception)) {
                /** @noinspection PhpUnhandledExceptionInspection */
                throw $exception;
            }
        } else {
            $result = parent::get_option($key, $empty_value);
        }

        return $result;
    }

    public function get_instance_id()
    {
        // A hack to prevent Woocommerce 2.6 from skipping global method instance
        // rates in WC_Shipping::calculate_shipping_for_package()
        return (method_exists('parent', 'get_instance_id') ? parent::get_instance_id() : $this->instance_id) ?: -1;
    }

    private function loadRule(IReader $reader, $ruleData = null)
    {
        if (!isset($ruleData)) {
            $ruleData = $this->get_option('rule');
            if ($ruleData) {
                $ruleData = json_decode($ruleData, true);
            }
        }

        if (!$ruleData) {
            $ruleData = array();
        }

        /** @var IRule $rule */
        $rule = $reader->read('rule', $ruleData);

        return $rule;
    }

    static private function initGlobalRegistry($lazy = true)
    {
        $settings = Units::fromPrecisions(
            pow(10, wc_get_price_decimals()),
            1000,
            1000
        );
        
        $globalRegistry = new GlobalRegistry($settings, $lazy);

        $globalRegistry->mappers->register('shipping_method_calculator', function() {
            return new ShippingMethodCalculatorMapper(new ShippingMethodLoader());
        });

        return $globalRegistry;
    }
}